namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class InventoryItem extends Model
{
    use HasFactory;

    protected $fillable = [
        'product_id',
        'use_by_date',
        'wholesale_price',
        'minimum_order_quantity',
        'available_quantity',
        'dutch_auction_enabled',
        'dutch_initial_price',
        'dutch_floor_price',
        'dutch_decrement_type',
        'dutch_decrement_value',
        'dutch_decrement_frequency_days',
        'current_price',
    ];

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function cartItems()
    {
        return $this->hasMany(CartItem::class, 'product_id');
    }

    public function orderItems()
    {
        return $this->hasMany(OrderItem::class, 'product_id');
    }

    public function calculateDutchPrice()
    {
        if (!$this->dutch_auction_enabled) {
            return $this->current_price;
        }

        $daysElapsed = now()->diffInDays($this->created_at);
        $decrements = floor($daysElapsed / $this->dutch_decrement_frequency_days);

        if ($this->dutch_decrement_type === 'amount') {
            $newPrice = $this->dutch_initial_price - ($decrements * $this->dutch_decrement_value);
        } elseif ($this->dutch_decrement_type === 'percentage') {
            $newPrice = $this->dutch_initial_price * pow((1 - $this->dutch_decrement_value / 100), $decrements);
        } else {
            $newPrice = $this->dutch_initial_price;
        }

        return max($newPrice, $this->dutch_floor_price);
    }
}